/**
* \file: utils.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author:  Veeraiyan Chidambaram /RBEI/ECF3/ veeraiyan.chidambaram@in.bosch.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include "Common.h"
#include "utils/Utils.h"
#include <algorithm>

using namespace std;

namespace adit { namespace carplay
{

CFLBooleanRef CFBooleanCreateFromInt(uint64_t inBoolean)
{
    CFLBooleanRef outBoolean = kCFBooleanFalse;
    if (inBoolean !=0)
        outBoolean = kCFBooleanTrue;

    CFRetain(outBoolean);

    return outBoolean;
}

CFNumberRef CFNumberCreateFromStdInt(uint64_t inInt)
{
    return (CFNumberCreateInt64(inInt));
}

CFStringRef CFStringCreateFromStdString(const string& inString)
{
    return(CFStringCreateWithCString(nullptr, inString.c_str(), kCFStringEncodingUTF8));
}

CFMutableArrayRef CFArrayCreateMutableFromStdList(const list<string>& inList)
{
    CFMutableArrayRef elementArray = CFArrayCreateMutable(nullptr, 0, &kCFTypeArrayCallBacks);
    if (elementArray != nullptr)
    {
        for (auto elementString : inList)
        {
            CFStringRef CFStr = CFStringCreateWithCString(nullptr, elementString.c_str(), kCFStringEncodingUTF8);
            if (CFStr == nullptr)
            {
                /*
                 * CFLArrayRemoveAllValues frees both CFStr and Mutable array
                 * and assign null to the elementArray
                 */
                CFLArrayRemoveAllValues(elementArray);
                break;
            }
            CFArrayAppendValue(elementArray, CFStr);
        }
    }

    return elementArray;
}

CFMutableDictionaryRef CFDictionaryFromStdMap(const map<string, string>& inMap,
        const CFFromStdPair* inPairs, uint32_t inNum, const string& inLog)
{
    string debugOutput = "";
    map<string, string> map = inMap;

    CFMutableDictionaryRef dictionary = CFDictionaryCreateMutable(nullptr, 0,
            &kCFTypeDictionaryKeyCallBacks,
            &kCFTypeDictionaryValueCallBacks);
    dipo_return_value_on_null(dipo, dictionary, nullptr);

    for (uint32_t i = 0; i < inNum; i++)
    {
        auto iterator = map.begin();
        if (map.end() != (iterator = map.find(inPairs[i].std)))
        {
            if (inPairs[i].type == CFFromStdType_Number)
            {
                // add as number
                int64_t number = strtol(iterator->second.c_str(), nullptr, 0);
                if (number >= LONG_MAX || number <= LONG_MIN)
                {
                    LOG_ERROR((dipo, "%s: invalid value %s for key %s",
                            inLog.c_str(), iterator->second.c_str(), iterator->first.c_str()));
                    number = 0;
                }
                CFDictionarySetNumber(dictionary, inPairs[i].cf,
                        kCFNumberSInt64Type, &number);
                debugOutput += string(" ") + iterator->first + string("=") +
                        std::to_string(number);
            }
            else
            {
                // add as string
                CFDictionarySetCString(dictionary, inPairs[i].cf, iterator->second.c_str(),
                        kSizeCString);
                debugOutput += string(" ") + iterator->first + string("=") + iterator->second;
            }

            map.erase(iterator);
        }
    }

    // report all remaining keys as invalid
    for (auto remainingPair : map)
    {
        LOG_ERROR((dipo, "%s: invalid key %s", inLog.c_str(), remainingPair.first.c_str()));
    }

    // missing space after '%s:' is on purpose
    LOGD_DEBUG((dipo, "%s:%s", inLog.c_str(), debugOutput.c_str()));
    return dictionary;
}

vector<string> SplitString(const string& inValue, const char inSeparators[])
{
    vector<string> result;

    string::size_type current = 0;
    string::size_type found;
    while (string::npos != (found = inValue.find_first_of(inSeparators, current)))
    {
        result.emplace_back(inValue, current, found - current);
        current = found + 1;
    }

    result.emplace_back(inValue, current);
    return result;
}

string TrimString(const string& inValue)
{
    string::size_type begin = inValue.find_first_not_of(" \t");
    if (begin == string::npos)
        begin = 0;
    string::size_type length = inValue.find_last_not_of(" \t");
    if (length == string::npos)
        length = inValue.length() - 1;
    else
        length = length - begin + 1;
    return inValue.substr(begin, length);
}

map<string, string> ParseMap(const string& inValue, bool inLowerCase, const string& inLog)
{
    map<string, string> dictionary;

    // split the string into 'xx=yy' key-value pairs
    vector<string> keyValuePairs = SplitString(inValue, " \t,");
    for (auto keyValuePair : keyValuePairs)
    {
        // split into key and value
        vector<string> keyValueVector = SplitString(TrimString(keyValuePair), "=");

        // in case we have multiple or no '='
        if (keyValueVector.size() != 2)
        {
            LOG_ERROR((dipo, "%s: could not parse %s as key-value pair",
                    inLog.c_str(), TrimString(keyValuePair).c_str()));
        }
        // in case we have a clear key-value pair
        else
        {
            string key = TrimString(keyValueVector[0]);
            if (inLowerCase)
            {
                std::transform(key.begin(), key.end(), key.begin(),
                        std::ptr_fun<int, int>(std::tolower));
            }

            string value = TrimString(keyValueVector[1]);

            // insert into result map
            dictionary.insert(pair<string, string>(key, value));
        }
    }

    return dictionary;
}

uint64_t GetCurrentTimeNano()
{
    timespec now;
    clock_gettime(CLOCK_MONOTONIC, &now);

    uint64_t seconds = (uint64_t)now.tv_sec * 1000000000;
    uint64_t nanoseconds = now.tv_nsec;

    return seconds + nanoseconds;
}

} } /* namespace adit { namespace carplay*/

